<?php

namespace App\Livewire\Pekerjaan;

use App\Models\Pekerjaan;
use App\Models\BadanUsaha;
use App\Models\TenagaAhli;
use App\Models\PaketPekerjaan;
use Livewire\Component;
use Livewire\Attributes\Rule;
use Livewire\Attributes\On;

class Form extends Component
{
    public $showModal = false;

    #[Rule('required|string|max:150')]
    public $nama = '';

    #[Rule('required|exists:badan_usaha,id')]
    public $penyedia_id = '';

    #[Rule('required|date')]
    public $tanggal_mulai = '';

    #[Rule('required|date|after_or_equal:tanggal_mulai')]
    public $tanggal_selesai = '';

    #[Rule('required|array|min:1')]
    public $tenaga_ahli = [];

    #[Rule('nullable|exists:paket_pekerjaans,id')]
    public $paket_pekerjaan_id = '';

    #[Rule('nullable|string')]
    public $deskripsi = '';

    #[Rule('nullable|string|max:150')]
    public $lokasi = '';

    public $editingId = null;

    #[On('form-show')]
    public function show($pekerjaanId = null)
    {
        $this->resetForm();
        $this->showModal = true;

        if ($pekerjaanId) {
            $this->editingId = $pekerjaanId;
            $pekerjaan = Pekerjaan::with('tenagaAhli')->find($pekerjaanId);

            $this->nama = $pekerjaan->nama;
            $this->penyedia_id = $pekerjaan->penyedia_id;
            $this->tanggal_mulai = $pekerjaan->tanggal_mulai->format('Y-m-d');
            $this->tanggal_selesai = $pekerjaan->tanggal_selesai->format('Y-m-d');
            $this->paket_pekerjaan_id = $pekerjaan->paket_pekerjaan_id;
            $this->deskripsi = $pekerjaan->deskripsi;
            $this->lokasi = $pekerjaan->lokasi;
            $this->tenaga_ahli = $pekerjaan->tenagaAhli->pluck('id')->toArray();
        }
    }

    public function save()
    {
        $this->validate();

        $data = [
            'nama' => $this->nama,
            'penyedia_id' => $this->penyedia_id,
            'tanggal_mulai' => $this->tanggal_mulai,
            'tanggal_selesai' => $this->tanggal_selesai,
            'paket_pekerjaan_id' => $this->paket_pekerjaan_id ?: null,
            'deskripsi' => $this->deskripsi,
            'lokasi' => $this->lokasi,
        ];

        if ($this->editingId) {
            $pekerjaan = Pekerjaan::find($this->editingId);
            $pekerjaan->update($data);
        } else {
            $pekerjaan = Pekerjaan::create($data);
        }

        $pekerjaan->tenagaAhli()->sync($this->tenaga_ahli);

        $this->dispatch('item-saved');
        $this->closeModal();
    }

    public function resetForm()
    {
        $this->reset([
            'nama', 'penyedia_id', 'tanggal_mulai', 'tanggal_selesai',
            'tenaga_ahli', 'paket_pekerjaan_id', 'deskripsi', 'lokasi', 'editingId'
        ]);
    }

    public function closeModal()
    {
        $this->showModal = false;
        $this->resetForm();
    }

    public function render()
    {
        $badanUsahas = BadanUsaha::orderBy('nama_badan_usaha')->get();
        $tenagaAhlis = TenagaAhli::orderBy('nama_tenaga_ahli')->get();
        $paketPekerjaans = PaketPekerjaan::orderBy('nama_paket')->get();

        return view('livewire.pekerjaan.form', [
            'badanUsahas' => $badanUsahas,
            'tenagaAhlis' => $tenagaAhlis,
            'paketPekerjaans' => $paketPekerjaans
        ]);
    }
}
